// Copyright (c) 2022 DIRB Co. Ltd.
//
// Use authorized under the MIT license.

#include "main.h"
#include "led.h"
#include <xc.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <util/delay.h>

uint8_t self_id = 0;        // holds stored self ID
uint8_t led_func;           // holds actual function number
uint8_t led_value;          // holds stored dim value
uint8_t led_delay;          // holds stored fading delay
uint8_t valueLED = 0;       // holds actual dim value
int16_t stepsLED = 0;       // holds actual fading steps
bool do_fade = false;       // fading active 
uint8_t icount = 0;         // fading isr counter 

// ---------------------------------------------------------------------------
// execute LED function based on MIN payload 
//
void led_function(uint8_t *function, uint8_t *value, uint8_t *delay) {
    
    led_func = *function;
    if (*delay == 0) (*delay)++;
    switch(led_func) {
        case LED_ON:
            valueLED = *value;
            if (pwmLED <= valueLED)
                stepsLED = 255 / *delay;
            else
                stepsLED = - (255 / *delay);                
            do_fade = true;
            break;
        case LED_ON_STORED:
            valueLED = led_value;
            if (pwmLED <= valueLED)
                stepsLED = 255 / *delay;
            else
                stepsLED = - (255 / *delay); 
            *value = led_value;
            *delay = led_delay;
            do_fade = true;
            break;
        case LED_OFF:
            valueLED = 0;
            stepsLED = - (255 / *delay);
            do_fade = true;
            break;
        case LED_OFF_STORED:
            valueLED = 0;        
            stepsLED = - (255 / led_delay);
            *value = 0;
            *delay = led_delay;
            do_fade = true;
            break;
        case LED_SET_STORED:
            led_value = *value;
            led_delay = *delay;
            eeprom_write_byte((uint8_t*)EE_VALUE, led_value);
            eeprom_write_byte((uint8_t*)EE_DELAY, led_delay);
            break;
        case LED_GET_STORED:
            *value = led_value;
            *delay = led_delay;
            break;
        case LED_STATUS:
            *value = pwmLED;
            *delay = eeprom_read_byte((uint8_t*)EE_INIT);
            break;
        case LED_STAT_ONOFF:
            if (*value == 0) {
                stat_led_off();
            } else {
                stat_led_on();
            }
            break;            
        default:
            *function = LED_ERROR;
            *value = LED_UNKNOWN_FUNC;
            *delay = 0;
    }
}

// ---------------------------------------------------------------------------
// init led related entities
//
void init_led(void) {
    
    self_id = eeprom_read_byte((uint8_t*)EE_SELF_ID);
    led_value = eeprom_read_byte((uint8_t*)EE_VALUE);
    led_delay = eeprom_read_byte((uint8_t*)EE_DELAY);
    flash_led(self_id);
    pwmLED = 0;
}


// ---------------------------------------------------------------------------
// flash LED to show self ID 
//
void flash_led(uint8_t id) {
      
    uint8_t i;  
    
    for (i=0;i<id;i++) {
        pwmLED = 1;
        stat_led_on();
        _delay_ms(200);
        pwmLED = 0;
        stat_led_off();
        _delay_ms(200);
    }
}


// ---------------------------------------------------------------------------
// get own ID
//
uint8_t get_self_id(void) {
    
    return self_id;
}


// ---------------------------------------------------------------------------
// turn status LED on
//
void stat_led_on(void) {

    PORTB |= (1<<PORTB0);
}


// ---------------------------------------------------------------------------
// turn status LERD off
//
void stat_led_off(void) {

    PORTB &= ~(1<<PORTB0);
}


// ---------------------------------------------------------------------------
// Interrupt service routine for LED PWM timer
// every 2.049ms (488Hz)
//
ISR(TIMER1_OVF_vect) {

    int16_t n;
    uint8_t sreg;
    
    sreg = SREG;    // save state of interrupt
    icount +=1;     // interrupt every 2.049ms (488Hz)
    
    // LED steps in 50ms
    if (icount == FADING_TIME) {
        // ramp up/down LED if requested
        if (do_fade) {
            if (stepsLED > 0) {
                if (((int16_t)pwmLED + stepsLED) > valueLED) {
                    n = valueLED - pwmLED;
                    do_fade = false;
                } else {
                    n = stepsLED;
                }
            } else {
                if (((int16_t)pwmLED + stepsLED) < valueLED) {
                    n = -(pwmLED - valueLED);
                    do_fade = false;
                } else {
                    n = stepsLED;
                }                            
            }
            pwmLED += n;
        }
        icount = 0;
    }
    SREG = sreg;  // restore state of interrupt
}