
#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdbool.h>
#include "main.h"
#include "SoftwareSerial.h"    

// ---------------------------------------------------------------------------
// Hardcoded TX/RX
//
volatile uint8_t *serddr  = 0;
volatile uint8_t *serport = 0;
volatile uint8_t *serpin  = 0;
uint8_t rxpin             = 0;
uint8_t txpin             = 0;

// ---------------------------------------------------------------------------
// Globals
//
uint16_t _rx_delay_centering = 0;
uint16_t _rx_delay_intrabit  = 0;
uint16_t _rx_delay_stopbit   = 0;
uint16_t _tx_delay           = 0;

bool _buffer_rx_overflow = false;

// static data
static char _receive_buffer[_SS_MAX_RX_BUFF];
static volatile uint8_t _receive_buffer_tail;
static volatile uint8_t _receive_buffer_head;

// private static method for timing
static inline void tunedDelay(uint16_t delay);
const int XMIT_START_ADJUSTMENT = 4;

// ===========================================================================
// Private methods
#define rx_pin_read() (*serpin & _BV(rxpin))

// ---------------------------------------------------------------------------
// delay in us
//
inline void tunedDelay(uint16_t delay) {
    
    uint8_t tmp = 0;

    asm volatile("sbiw    %0, 0x01 \n\t"
                 "ldi %1, 0xFF \n\t"
                 "cpi %A0, 0xFF \n\t"
                 "cpc %B0, %1 \n\t"
                 "brne .-10 \n\t"
                 : "+w" (delay), "+a" (tmp)
                 : "0" (delay)
                 );
}

// ---------------------------------------------------------------------------
// ISR to receive byte
// triggered by start bit (falling edge))
//
ISR(PCINT0_vect) {

    uint8_t sreg;
    
    sreg = SREG;   // save state of interrupt
    cli();         // disable interrupts
    
    uint8_t data = 0, i; 
    
    // If RX line is high, then we don't see any start bit
    // so interrupt is probably not for us
    if ( !rx_pin_read() ) {
        // Wait approximately 1/2 of a bit width to "center" the sample
        tunedDelay(_rx_delay_centering);

        // Read each of the 8 bits
        for (i = 0x1; i; i <<= 1) {
          tunedDelay(_rx_delay_intrabit);
          if (rx_pin_read())
            data |= i;
          else // else clause added to ensure function timing is ~balanced
            data &= ~i;
        }

        // skip the stop bit
        tunedDelay(_rx_delay_stopbit);

        // if buffer full, set the overflow flag and return
        if (((_receive_buffer_tail + 1) & _SS_RX_BUFF_MASK) != _receive_buffer_head) {  // circular buffer
          // save new data in buffer: tail points to where byte goes
          _receive_buffer[_receive_buffer_tail] = data; // save new byte
          _receive_buffer_tail = (_receive_buffer_tail + 1) & _SS_RX_BUFF_MASK;  // circular buffer
        } else {
          _buffer_rx_overflow = true;
        }
    }
    SREG = sreg;  // restore state of interrupt
}

// ===========================================================================
// Public methods
//

// ---------------------------------------------------------------------------
// Init software serial interface
//
void softSerialInit( volatile uint8_t *ddr, volatile uint8_t *port,
                     volatile uint8_t *pin, uint8_t rx, uint8_t tx )  {
    
    serddr  = ddr;
    serport = port;
    serpin  = pin;
    rxpin   = rx;
    txpin   = tx;
}
   
// ---------------------------------------------------------------------------
// Start software serial interface at given baud rate
//
void softSerialBegin(long baud) {

    _receive_buffer_head = _receive_buffer_tail = 0;
    _buffer_rx_overflow = false;

    *serddr  |= _BV(txpin);   // set TX for output
    *serport |= _BV(txpin);   // assumes no inverse logic

    *serddr  &= ~_BV(rxpin);  // set RX for input
    *serport |= _BV(rxpin);   // assumes no inverse logic  
    
    _tx_delay = F_CPU/(7 * baud) - 6;
    _rx_delay_stopbit = _tx_delay + 3;
    _rx_delay_intrabit = _rx_delay_stopbit;
    _rx_delay_centering = _tx_delay/2 - 5;

    // General Interrupt Mask Register
    // GIMSK = [?:INT0:PCIE:?:?:?:?:?]
    // set PCIE to enable pin change interrupts
    GIMSK |= (1<<PCIE);
    // Pin Change Mask Register
    // PCMSK = [-:-:PCINT5:PCINT4:PCINT3:PCINT2:PCINT1:PCINT0]
    // enable rxpin interrupts 
    PCMSK |= _BV(rxpin);
    
    tunedDelay(_tx_delay);
}


// ---------------------------------------------------------------------------
// Stop software serial interface 
//
void softSerialEnd() {
    
    PCMSK = 0;     // disable interrupt 
}


// ---------------------------------------------------------------------------
// Read data byte from receive buffer
//
int16_t softSerialRead() {

    uint8_t data;

    // no data in buffer ?
    if (_receive_buffer_head == _receive_buffer_tail)
        return -1;

    // Read from "head"
    data = _receive_buffer[_receive_buffer_head]; // grab next byte
    _receive_buffer_head = (_receive_buffer_head + 1) & _SS_RX_BUFF_MASK; // circular buffer
    return (int16_t)data;
}

// ---------------------------------------------------------------------------
// Return number of data bytes in receive buffer
//
uint8_t softSerialAvailable() {
    
    return ((_receive_buffer_tail + _SS_MAX_RX_BUFF - _receive_buffer_head) & _SS_RX_BUFF_MASK); // circular buffer
}

// ---------------------------------------------------------------------------
// returns overflow status of RX buffer and reset status
//
bool softSerialOverflow(void) {

    bool status = _buffer_rx_overflow;
    _buffer_rx_overflow = false;
    return status;
}

// ---------------------------------------------------------------------------
// transmit data byte
//
void softSerialWrite(uint8_t data) {
    
    byte mask;
    
    uint8_t oldSREG = SREG; // store interrupt flag
    cli();                  // turn off interrupts for a clean txmit
    
    PORTB |= (1<<PORTB2);   // set direction of RS485 driver to Tx

    // Write start bit
    *serport &= ~_BV(txpin); // tx pin low
    tunedDelay(_tx_delay + XMIT_START_ADJUSTMENT);

    // Write each of the 8 bits
    for (mask = 0x01; mask; mask <<= 1) {
      if (data & mask) { // choose bit
        *serport |= _BV(txpin); // tx pin high, send 1
      }
      else {
        *serport &= ~_BV(txpin); // tx pin low, send 0
      }
      tunedDelay(_tx_delay);
    }
    
    // Write stop bit
    *serport |= _BV(txpin);     // tx pin high, restore pin to natural state
    tunedDelay(_tx_delay);

    PORTB &= ~(1<<PORTB2);      // set direction of RS485 driver back to Rx    
    SREG = oldSREG;             // turn interrupts back on
}

// ---------------------------------------------------------------------------
// flush receive buffer
//
void softSerialFlush() {
  
    uint8_t oldSREG = SREG; // store interrupt flag
    cli();
    _receive_buffer_head = _receive_buffer_tail = 0;
    SREG = oldSREG; // restore interrupt flag
}

// ---------------------------------------------------------------------------
// peek receive buffer
//
int16_t softSerialPeek() {
    // Empty buffer?
    if (_receive_buffer_head == _receive_buffer_tail)
        return -1;

    // Read from "head"
    return (int16_t)_receive_buffer[_receive_buffer_head];
}


// ---------------------------------------------------------------------------
// read 'buf_len' bytes from receive buffer
//
uint8_t softSerialReadBytes(uint8_t *buf, uint8_t buf_len) {
    
    uint8_t i;
    
    if (buf_len > softSerialAvailable()) buf_len = softSerialAvailable();
    for(i = 0; i < buf_len; i++) {
        buf[i] = (uint8_t)softSerialRead();
    }
    return buf_len;
}
