// Copyright (c) 2022 DIRB Co. Ltd.
//
// Use authorized under the MIT license.

/* 
 *  RS485 LED Controller
 *  ===============================================================
 *  The connections to the ATTiny are as follows:
 *  ATTiny  USED     I/O    Pin Function                              
 *  Pin 1   dW       I      PB5 PCINT5/ADC0/NRES/dW                 
 *  Pin 2   TxD      O      PB3 PCINT3/ADC3/USB+/NOC1B/CLKI            
 *  Pin 3   PWM LED  O      PB4 PCINT4/ADC2/USB-/OC1B/CLKO             
 *  Pin 4   GND
 *  Pin 5   DIR      O      PB0 PCINT0/AIN0/MOSI/SDA/DI/OC0A/NOC1A     
 *  Pin 6   RxD      I      PB1 PCINT1/AIN1/MISO/DO/OC0B/OC1A          
 *  Pin 7   LED      O      PB2 PCINT2/ADC1/SCK/SCL/USCK/INT0          
 *  Pin 8   +Vcc
 */

#include "main.h"
#include "min.h"
#include "led.h"
#include "SoftwareSerial.h"
#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <xc.h>
#include <avr/eeprom.h>
#include <stdbool.h>

#ifdef SET_ID
// EEPROM preset
uint8_t EEMEM ID[4] = {INIT_ID, LED_ID, 0x00, 0x01};
#endif        

// ATtiny85 Configuration Bit Settings (only via ISP programming I/F)
FUSES = {
	//.low = 0xE2, // LOW {SUT_CKSEL=INTRCOSC_8MHZ_6CK_14CK_64MS, CKOUT=CLEAR, CKDIV8=CLEAR}
    .low = 0x62, // LOW {SUT_CKSEL=INTRCOSC_8MHZ_6CK_14CK_64MS, CKOUT=CLEAR, CKDIV8=SET}
	.high = 0x9F, // HIGH {BODLEVEL=DISABLED, EESAVE=CLEAR, WDTON=CLEAR, SPIEN=SET, DWEN=SET, RSTDISBL=CLEAR}
	.extended = 0xFF, // EXTENDED {SELFPRGEN=CLEAR}
};
LOCKBITS = 0xFF; // {LB=NO_LOCK}

struct min_context min_ctx;         // min protocol context
        
//////////////////////////// MIN CALLBACKS ///////////////////////////////////
//
// Tell MIN how much space there is to write to the serial port. This is used
// inside MIN to decide whether to bother sending a frame or not.
uint16_t min_tx_space(uint8_t port)
{
  return MAX_BYTES_PER_FRAME;
}

// ---------------------------------------------------------------------------
// Send a character on the designated port.
void min_tx_byte(uint8_t port, uint8_t byte)
{
  // Ignore 'port' because we have just one context.
  softSerialWrite(byte);  
}

// ---------------------------------------------------------------------------
void min_tx_start(uint8_t port) {}
// ---------------------------------------------------------------------------
void min_tx_finished(uint8_t port) {}

// ---------------------------------------------------------------------------
// Handle the reception of a MIN frame. This is the main interface to MIN for receiving
// frames. It's called whenever a valid frame has been received (for transport layer frames
// duplicates will have been eliminated).
void min_application_handler(uint8_t min_id, uint8_t *min_payload, uint8_t len_payload, uint8_t port)
{
  // only correct payload size will be parsed  
  if (len_payload == LED_PAYLOAD_SIZE)  {
    // check if this led light is targeted or it's a global command  
    if ((min_id == get_self_id()) | (min_id == LED_GLOBAL))  { 
        // parse payload and execute function
        led_function(min_payload, min_payload+1, min_payload+2);
        if ((min_id != LED_GLOBAL)) {
            min_id |= LED_RESPONSE_BIT;
            min_send_frame(&min_ctx, min_id, min_payload, len_payload);
        }
    }
  } else {
    // wrong payload! send error message back
    if ((min_id != LED_GLOBAL)) {
        min_id |= LED_RESPONSE_BIT;
        *(min_payload) = LED_ERROR;
        *(min_payload+1) = LED_ERROR_PAYLOAD_SIZE;
        *(min_payload+2) = len_payload; 
        min_send_frame(&min_ctx, min_id, min_payload, LED_PAYLOAD_SIZE);
    }      
  }
}

// ---------------------------------------------------------------------------
// initalize ATtiny85 
//
void init_attiny(void)
{
    // Set the CPU prescaler for 8 MHz
    CLKPR = (1 << CLKPCE); 
    CLKPR = (0x00);

    // **** PWM for LED (488Hz) ***
    //
    // Control Register for Timer/Counter-1 
    // TCCR1 = [CTC1:PWM1A:COM1A1:COM1A0:CS13:CS12:CS11:CS10]
    // bit PWM1A remains clear, which prevents Timer/Counter-1 from using pin OC1A (which is shared with OC0B)
    // bits COM1A0 and COM1A1 remain clear, which also prevents Timer/Counter-1 from using pin OC1A (see PWM1A above)
    // sets bit CS10-12 which tells Timer/Counter-1 to pre-scale CK/64
    TCCR1 = (0<<PWM1A) | (0<<COM1A0) | (1<<CS12) | (1<CS11) | (1<<CS10);
    
    // General Control Register for Timer/Counter-1 
    // GTCCR = [TSM:PWM1B:COM1B1:COM1B0:FOC1B:FOC1A:PSR1:PSR0]
    // sets bit PWM1B which enables the use of OC1B (since we disabled using OC1A in TCCR1)
    // sets bit COM1B1 and leaves COM1B0 clear, which (when in PWM mode) clears OC1B on compare-match, and sets at BOTTOM
    GTCCR = (1<<PWM1B) | (2<<COM1B0);    
    OCR1B = 0x00;  // start with 0%
  
    // TIMSK ? Timer/Counter Interrupt Mask Register
    // TIMSK = [?:OCIE1A:OCIE1B:OCIE0A:OCIE0B:TOIE1:TOIE1?] 
    // set TOIE1 which enables Timer/Counter-1 overflow interrupt
    TIMSK = (1<<TOIE1);
    
    // Port B Data Register
    // DDPORTBB = [?:?:PORTB5:PORTB4:PORTB3:PORTB2:PORTB1:PORTB0]
    // set PB1,PB3 to high
    // set PB0,PB2,PB4 to low
    PORTB = (0<<PORTB5) | (0<<PORTB4) | (1<<PORTB3) | (0<<PORTB2) | (1<<PORTB1) | (0<<PORTB0);

    // Port B Data Direction Register
    // DDRB = [?:?:DDB5:DDB4:DDB3:DDB2:DDB1:DDB0]
    // set PB0,PB2,PB3,PB4 to be output
    // set PB1 to be input
    DDRB = (0<<DDB5) | (1<<DDB4) | (1<<DDB3) | (1<<DDB2) | (0<<DDB1) | (1<<DDB0);
}


// ---------------------------------------------------------------------------
// MAIN program
//
int main (void)  { 
    
    // Initialize ATTINY85
    init_attiny();
    
    // setup software serial interface (RX=PB1, TX=PB3)
    softSerialInit(&DDRB,&PORTB,&PINB,1,3);
    softSerialBegin(9600);
    
    // Initialize LED and flash ID number
    init_led();
            
    // Initialize the single context. Since we are going to ignore the port value we could
    // use any value. But in a bigger program we would probably use it as an index.
    min_init_context(&min_ctx, 0);
    
    sei();
    // infinite loop
    while (1) {

        uint8_t buf[32];
        uint8_t buf_len;

        // Read some bytes from the serial port...
        if(softSerialAvailable() > 0) {
            buf_len = softSerialReadBytes(buf, MAX_BYTES_PER_FRAME);
            // .. and push them into MIN. It doesn't matter if the bytes are read in one by
            // one or in a chunk (other than for efficiency) so this can match the way in which
            // serial handling is done (e.g. in some systems the serial port hardware register could
            // be polled and a byte pushed into MIN as it arrives).
            min_poll(&min_ctx, buf, buf_len);   
        }       
    }
    return 1;
}






