#include <SPI.h>
#include <stdio.h>
#include <string.h>

// mode states for dab module
#define NONE  0
#define FM    1
#define DAB   2

#define TEST_FM_FREQ  88000     // FM freuency used for module test 

// implentartion of switch construct with strings
#ifndef SWITCH_CASE_INIT
#define SWITCH_CASE_INIT
  #define SWITCH(X) for (char* __switch_p__ = X, __switch_next__=1 ; __switch_p__ ; __switch_p__=0, __switch_next__=1) { {
    #define CASE(X)     } if (!__switch_next__ || !(__switch_next__ = strcmp(__switch_p__, X))) {
  #define DEFAULT       } {
  #define END     }}
#endif

// time structure for local time
typedef struct _dab_time
{
  uint16_t  year;
  uint8_t   months;
  uint8_t   days;
  uint8_t   hours;
  uint8_t   minutes;
  uint8_t   seconds;
} dab_time_t;

//SPI Ports connected to dab module
const byte dabpin_ss = 8;      
const byte dabpin_clk = 13;
const byte dabpin_miso = 12;
const byte dabpin_mosi = 11;
//miec ports connected to dab module
const byte dabpin_intb = 2;
const byte dabpin_resetb = 7;

// variables used by si4684_func.c
extern uint8_t   ver_major;         // firmware version major
extern uint8_t   ver_minor;         // firmware version minor
extern uint8_t   ver_build;         // firmware version build
extern uint16_t  freq;              // FM frequency
extern uint32_t  dabfreq;           // DAB frequency
extern int8_t    rssi;              // RSSI [dBuV]
extern int8_t    snr;               // SNI [dB]
extern char      fm_ps[];           // FM Program Service Name
extern char      service_data[];    // DAB/FM service data
extern uint16_t  fm_pi;             // FM program id
extern uint8_t   fm_pty;            // FM program type 

// local variables
uint8_t mode = 0;                   // mode dab module [NONE,FM,DAB]
uint8_t vol = 63;                   // audio volume 
uint8_t mute = 0;                   // audio mute
uint8_t service_no = 0;             // selected DAB service number
dab_time_t localtime;               // local time from DAB or FM
byte rxindex = 0;                   // index in UART receive buffer
char rxdata[20];                    // UART receive buffer

//-----------------------------------------------------------------------------------------
// Shows Comnand list
//-----------------------------------------------------------------------------------------
void show_commands(void)
{
  Serial.print(F("\n\nCOMMANDS:\n")); 
  Serial.print(F(" reset            -> Reset Si4684\n"));
  Serial.print(F(" test <fm/dab>    -> Test module (FM/DAB Mode)\n"));
  Serial.print(F(" fm               -> Set FM Mode\n"));
  Serial.print(F(" dab              -> Set DAB Mode\n"));
  Serial.print(F(" help             -> display this command list\n"));

  Serial.print(F(" DAB Mode:\n"));
  Serial.print(F("  scan            -> scan for available DAB ensembles\n"));
  Serial.print(F("  tune <n>        -> tune to frequency index\n"));
  Serial.print(F("  service <n>     -> tune to service ID\n"));
  Serial.print(F("  info            -> display the ensemble info\n"));
  Serial.print(F("  time            -> display the current DAB time\n"));
  Serial.print(F("  version         -> display DAB firmware version\n"));
  Serial.print(F("  vol <n>         -> set volume 0-63\n"));
  Serial.print(F("  mute <n>        -> mute 0=off,1=left,2=right,3=all\n"));

  Serial.print(F(" FM Mode:\n"));
  Serial.print(F("  scan            -> scan available FM stations\n"));
  Serial.print(F("  tune <n>        -> tune to frequency in kHz\n"));
  Serial.print(F("  seek <up/down>  -> seek to next station (up/down)\n"));
  Serial.print(F("  info            -> display signal quality and RDS data\n"));
  Serial.print(F("  version         -> display FM firmware version\n"));
  Serial.print(F("  vol <n>         -> set volume 0-63\n"));
  Serial.print(F("  mute <n>        -> mute 0=off,1=left,2=right,3=all\n"));

  Serial.print(F("\n"));
}

//-----------------------------------------------------------------------------------------
// parse command received from UART
//-----------------------------------------------------------------------------------------
void parse_command(char *command)
{
  char buffer[16];

  char *cmd;
  cmd = strtok(command, " \r");

  SWITCH (cmd)
    CASE ("reset")
        mode = NONE;
        si468x_reset();
        break;
    CASE ("test")
        cmd = strtok(NULL, " \r");
        if(strcmp(cmd, "dab") == 0)
        {
          if (dab_module_test()) mode = DAB;
        }
        else if(strcmp(cmd, "fm") == 0)
        {
          if (fm_module_test(TEST_FM_FREQ)) mode = FM;
        }        
        break;
    CASE ("fm")          
        if (fm_init() == 0) mode = FM;
        break;        
    CASE ("dab")
        if (dab_init() == 0) mode = DAB;
        break;
    CASE ("help")
        show_commands();
        break; 
    CASE ("version")
        if (mode != NONE) {
          si468x_get_func_info();
          Serial.print(F(" FW Ver:"));
          sprintf(buffer, "%u.%u.%u\n", ver_major, ver_minor, ver_build);
          Serial.print(buffer); 
        }
        break; 
    CASE ("scan")
        if (mode == DAB)
        {
          dab_scan();
        }
        if (mode == FM)
        {
          fm_scan();
        }
        break; 
    CASE ("seek")
        cmd = strtok(NULL, " \r");
        if (mode == FM)
        {
          if(strcmp(cmd, "up") == 0)
          {
            fm_seek(true, true);
          }
          if(strcmp(cmd, "down") == 0)
          {
            fm_seek(false, true);
          }   
        }     
        break;
    CASE ("info")
        if (mode == DAB)
        {
          dab_ensemble_info();
        }
        if (mode == FM)
        {
          fm_status();
        }
        break; 
    CASE ("tune")
        cmd = strtok(NULL, " \r");
        if (mode == DAB)
        {
          freq = (int)strtol(cmd, NULL, 10);
          service_no = 0;
          dab_tune(freq); 
          dab_ensemble_info();
        }
        if (mode == FM)
        {
          cmd[strlen(cmd)-1] = "\0";
          freq = (int)strtol(cmd, NULL, 10);
          if ((freq >= 8750) && (freq <= 10790))
          {
            fm_tune_freq(freq);
            fm_status();      
          }
          else
          {   
            Serial.print(F("Freq out of range 87.5 - 107.9MHz!\n"));
          }          
        }
        break; 
    CASE ("service")
        cmd = strtok(NULL, " \r");
        if (mode == DAB && (int)strtol(cmd, NULL, 10) < 16)
        {
          service_no = (int)strtol(cmd, NULL, 10);
          dab_set_service(service_no);
          si468x_set_volume(vol);
        }
        break;
    CASE ("vol")
        if (mode != NONE)
        {
          cmd = strtok(NULL, " \r");
          vol = (int)strtol(cmd, NULL, 10);
          si468x_set_volume(vol);
        }
        break;
    CASE ("mute")
        if (mode != NONE)
        {
          cmd = strtok(NULL, " \r");
          mute = (int)strtol(cmd, NULL, 10);
          si468x_mute(mute);
        }
        break;
    CASE ("time")
        if (mode == DAB)
        {
          char buffer[16];
          si468x_get_time(&localtime);
          sprintf(buffer,"%02u.%02u.%02u ", localtime.days,localtime.months,localtime.year);
          Serial.print(buffer);
          sprintf(buffer,"%02u:%02u\n", localtime.hours,localtime.minutes);
          Serial.print(buffer);
        }
        if (mode == FM)
        {
          sprintf(buffer,"%02u.%02u.%02u ", localtime.days,localtime.months,localtime.year);
          Serial.print(buffer);
          sprintf(buffer,"%02u:%02u\n", localtime.hours,localtime.minutes);
          Serial.print(buffer);
        }
        break;
    DEFAULT
        Serial.print(F("unknown command\n"));   
  END
  switch (mode)
  {
    case NONE:
        Serial.print(F("none>"));
        break;
    case FM:
        Serial.print(F("FM>"));
        break;
    case DAB:
        Serial.print(F("DAB>"));
        break;
  }
}

//-----------------------------------------------------------------------------------------
// FM call back function. Called if new RDS data available
// rdsdata 0x0001  new program name 
//         0x0004  new servie data
//         0x0010  new time update
//-----------------------------------------------------------------------------------------
void fm_callback(uint16_t rdsdata)
{
  // char buffer[66];

  // sprintf(buffer,"\n%02u.%02u.%04u ", localtime.days,localtime.months,localtime.year);
  // Serial.print(buffer);
  // sprintf(buffer,"%02u:%02u", localtime.hours,localtime.minutes);
  // Serial.print(buffer);
  // Serial.print(F(" PI="));
  // sprintf(buffer,"%u",fm_pi);
  // Serial.print(buffer);
  // Serial.print(F(" PTY="));
  // sprintf(buffer,"%u",fm_pty);
  // Serial.print(buffer);
  // Serial.print(F(" PS="));
  // sprintf(buffer,"%s",fm_ps);
  // Serial.print(buffer);
  // Serial.print(F(" SD="));
  // sprintf(buffer,"%s\n",service_data);
  // Serial.print(buffer);
}

//-----------------------------------------------------------------------------------------
// SETUP
//-----------------------------------------------------------------------------------------
void setup() {
  //Intitialise the UART
  Serial.begin(115200);
  while(!Serial);

  Serial.print(F("\n\nFM/DAB+ Module Test...\n")); 

  // enable SPI
  pinMode(dabpin_ss, OUTPUT);
  digitalWrite(dabpin_ss, HIGH);
  SPI.begin();

  // configure additional pins
  pinMode(dabpin_resetb, OUTPUT);
  digitalWrite(dabpin_resetb, HIGH);
  pinMode(dabpin_intb, INPUT_PULLUP);
  digitalWrite(dabpin_intb, HIGH);

  fm_set_callback(fm_callback);
  show_commands(); 
  mode = NONE;
  Serial.print(F("none>"));
}

//-----------------------------------------------------------------------------------------
// LOOP
//-----------------------------------------------------------------------------------------
void loop() 
{
  // put your main code here, to run repeatedly:
  if(digitalRead(dabpin_intb) == LOW)
  {
    if (mode == FM) fm_data_service();
    //if (mode == DAB) dab_data_service();  --> not implemented
  }
  // handle commands recied via UART
  if (Serial.available() > 0)
  {
    rxdata[rxindex] = Serial.read();
    if (rxdata[rxindex] == '\r')        //return
    {
      Serial.print(F("\n"));
      rxdata[rxindex] = '\0';

      parse_command(rxdata);            // parse command string
      rxindex = 0;
    }
    else if (rxdata[rxindex] == '\b')   //backspace
    {
      if (rxindex > 0)
      {
        Serial.print(F("\b \b"));
        rxindex--;
      }
    }
    else                                //other char
    {
      Serial.print(rxdata[rxindex]);
      rxindex++;
      if (rxindex >= 32)
      {
        rxindex = 0;
      }
    }
  }
}


//===============================================
// pin related functions used by si468x functions
//===============================================

//-----------------------------------------------------------------------------------------
// Send message via SPI to Si468x
//-----------------------------------------------------------------------------------------
void si468x_SpiMsg(unsigned char *data, uint32_t len)
{
  SPI.beginTransaction(SPISettings(2000000, MSBFIRST, SPI_MODE0));    //2MHz for starters...
  digitalWrite (dabpin_ss, LOW);
  SPI.transfer(data, len);
  digitalWrite (dabpin_ss, HIGH);
  SPI.endTransaction();
}

//-----------------------------------------------------------------------------------------
// set RESET pin at Si468x to low
//-----------------------------------------------------------------------------------------
void si468x_SetLevelRESETlow(void) 
{
  digitalWrite(dabpin_resetb, LOW);  //Reset LOW
}

//-----------------------------------------------------------------------------------------
// set RESET pin at Si468x to high
//-----------------------------------------------------------------------------------------
void si468x_SetLevelRESEThigh(void) 
{
  digitalWrite(dabpin_resetb, HIGH);  //Reset LOW
}

//-----------------------------------------------------------------------------------------
// get INTB level at Si468x
//-----------------------------------------------------------------------------------------
uint8_t si468x_GetLevelINTB(void)
{
  return(digitalRead(dabpin_intb));
}
