//===================================================================
// si468x FUNCTIONS FOR FM/DAB+
// DIRB
// V0.1 INITIAL VERSION
//
//===================================================================

/**
 * @file si468x_func.c
 * @author DIRB
 * @date 22 Jan 2021
 * @brief Low-level function of si468x FM/DAB+ 
 *  function to be defined externally 
 *  void si468x_SpiMsg(unsigned char *data, uint32_t len)
 *  void si468x_SetLevelRESETlow(void);
 *  void si468x_SetLevelRESEThigh(void);
 *  void si468x_GetLevelINTB(void);
 * @see http://tech.rbulla.de/doku.php?id=de:tech:dabmodule
 */

#include "si4684patch.h"
#include "si4684.h"

/**------------------------------------------------------------------
 * @brief Reset si468x
 * In RESET state analog and digital circuitry is disabled, and the 
 * VA, VCORE and VMEM power supplies are internally disconnected
 * @param none
 * @return Clear to send status
 */
uint8_t si468x_reset(void) 
{  
  si468x_SetLevelRESETlow();  //Reset LOW
  delay(100);
  si468x_SetLevelRESEThigh(); //Reset HIGH
  delay(100);  // wait min 5ms until Power Up Cmd
  //si468x_responseN(100);
  return(si468x_power_up());
}


/**------------------------------------------------------------------
 * @brief Power-up the device and set system settings
 * @param none
 * @return Clear to send status
 */
uint8_t si468x_power_up(void)
{
  spi_buf[0] = SI46XX_POWER_UP;
  spi_buf[1] = 0x00;   // CTSIEN[7] -> 0x80: Enable toggling host interrupt line when CTS is available.
  spi_buf[2] = 0x17;   // CLK_MODE[5:4], TR_SIZE[3:0]
  spi_buf[3] = 0x48;   // IBIAS[6:0]
  spi_buf[4] = 0x00;   // XTAL_FREQ[7:0]
  spi_buf[5] = 0xf8;   // XTAL_FREQ[15:8]
  spi_buf[6] = 0x24;   // XTAL_FREQ[23:16]
  spi_buf[7] = 0x01;   // XTAL_FREQ[31:24] --> 19.200MHz
  spi_buf[8] = 0x1F;   // CTUN[5:0]
  spi_buf[9] = 0x10;
  spi_buf[10] = 0x00;
  spi_buf[11] = 0x00;
  spi_buf[12] = 0x00;
  spi_buf[13] = 0x18;  // IBIAS_RUN[6:0]
  spi_buf[14] = 0x00;
  spi_buf[15] = 0x00;

  si468x_SpiMsg(spi_buf, 16);

  return(si468x_cts());
}


/**------------------------------------------------------------------
 * @brief Prepares the bootloader to receive a new image
 * @param none
 * @return Clear to send status
 */
uint8_t si468x_load_init(void)
{
  spi_buf[0] = SI46XX_LOAD_INIT;
  spi_buf[1] = 0x00;

  si468x_SpiMsg(spi_buf, 2);
  return(si468x_cts());
}


/**------------------------------------------------------------------
 * @brief Loads an image from HOST over command interface
 * @param none
 * @return Clear to send status
 */
uint8_t si468x_host_load(void)
{
  uint16_t index;
  uint16_t patchsize;
  uint16_t i;
  uint8_t result = 0;

  patchsize = sizeof(rom_patch_016);
  index = 0;

  while (index < patchsize)
  {
    spi_buf[0] = SI46XX_HOST_LOAD;
    spi_buf[1] = 0x00;
    spi_buf[2] = 0x00;
    spi_buf[3] = 0x00;
    for (i = 4; (i < SPI_BUFF_SIZE) && (index < patchsize); i++)
    {
      spi_buf[i] = pgm_read_byte_near(rom_patch_016 + index);
      index++;
    }
    si468x_SpiMsg(spi_buf, i);
    result |= si468x_cts();
  }
  return(result);
}


/**------------------------------------------------------------------
 * @brief Reports basic information about the device
 * @param none
 * @param[out] ChipRevision
 * @param[out] RomID
 * @param[out] PartNo
 * @return Clear to send status
 */
uint8_t si468x_get_part_info(void)
{
  uint8_t cmd_error;
  
  spi_buf[0] = SI46XX_GET_PART_INFO;
  spi_buf[1] = 0x00;
  
  si468x_SpiMsg(spi_buf, 2);
  if (cmd_error = si468x_cts() == 0)
  {
    si468x_responseN(10);
    chip_rev = spi_buf[5];
    rom_id = spi_buf[6];
    part_no = (spi_buf[10] << 8) + spi_buf[9];
    return(0);
  }
  return(cmd_error);
}


/**------------------------------------------------------------------
 * @brief Tunes the FM receiver to a frequency in 10 kHz steps.
 * @param freq frequency in KHz / 10 [16-bit]
 * @param[out] cmd_error  command error code, bit 7 = 1 : time oout
 * @return Command status
 */
uint8_t si468x_fm_tune_freq(uint16_t freq)
{
  uint32_t timeout;
    
  spi_buf[0] = SI46XX_FM_TUNE_FREQ;
  spi_buf[1] = 0x00;

  spi_buf[2] = freq & 0xFF;
  spi_buf[3] = (freq >> 8) & 0xFF;
  spi_buf[4] = 0x00;
  spi_buf[5] = 0x00;
  spi_buf[6] = 0x00;

  si468x_SpiMsg(spi_buf, 7);
  cmd_error = si468x_cts();

  timeout = 1000;
  do
  {
    delay(4);
    si468x_response();
    timeout--;
    if(timeout == 0)
    {
      cmd_error |= 0x80;
      break;
    }
  }
  while ((spi_buf[1] & 0x01) == 0); //STCINT
  return (cmd_error);
}


/**------------------------------------------------------------------
 * @brief Returns status information about the received signal quality.
 * @param none
 * @param[out] valid  Valid according to thresholds
 * @param[out] freq   Currently tuned frequency
 * @param[out] rssi   Received signal strength indicator in dBuV
 *v@param[out] snr    RF SNR indicator in dB
 * @return none
 */
void si468x_fm_rsq_status(void)
{
  spi_buf[0] = SI46XX_FM_RSQ_STATUS;
  spi_buf[1] = 0x00;
  si468x_SpiMsg(spi_buf, 2);

  si468x_cts();
  si468x_responseN(12);

  valid = (spi_buf[6] & 0x01) == 0x01 ? true : false;  
  freq = spi_buf[7] + ((uint16_t)spi_buf[8] << 8);;
  rssi = spi_buf[10];
  snr = spi_buf[11];
}


/**------------------------------------------------------------------
 * @brief Returns status information about the received signal quality.
 * @param seekup  0 : Seek down, 1 : Seek up
 * @param wrap  0 : Halt seek at band limit, 
 *              1 : When band limit is hit, continue seek from opposite band limit.
 * @param[out] cmd_error  command error code, bit 7 = 1 : time oout
 * @return none
 */
void si468x_fm_seek(uint8_t seekup, uint8_t wrap)
{
  uint32_t timeout;

  spi_buf[0] = SI46XX_FM_SEEK_START;
  spi_buf[1] = 0x00;;
  spi_buf[2] = ((seekup == 0) ? 0 : 2) | ((wrap == 0) ? 0 : 1);
  spi_buf[3] = 0x00;
  spi_buf[4] = 0x00;
  spi_buf[5] = 0x00;

  si468x_SpiMsg(spi_buf, 6);
  cmd_error = si468x_cts();

  timeout = 10000;
  do
  {
    delay(4);
    si468x_response();
    timeout--;
    if(timeout == 0)
    {
      cmd_error |= 0x80;
      break;
    }
  }
  while ((spi_buf[1] & 0x01) == 0); //STCINT
}


/**------------------------------------------------------------------
 * @brief Queries the status of RDS decoder and Fifo
 * @param none
 * @param[out] blockA,blockB,blockC,blockD  
 * @param[out] fm_pi, fm_pty  
 * @return none
 */
void si468x_get_fm_rds_status(void)
{
  spi_buf[0] = SI46XX_FM_RDS_STATUS;
  spi_buf[1] = 0x01;
  si468x_SpiMsg(spi_buf, 2);

  si468x_cts();
  si468x_responseN(20);

  if((spi_buf[6] & 0x08) == 0x08)
    fm_pi = spi_buf[9] + ((uint16_t)spi_buf[10] << 8); // Program ID
  if((spi_buf[6] & 0x10) == 0x10)
    fm_pty = spi_buf[7] & 0x1f;                        // Program Type 

  blockA = spi_buf[13] + ((uint16_t)spi_buf[14] << 8);
  blockB = spi_buf[15] + ((uint16_t)spi_buf[16] << 8);
  blockC = spi_buf[17] + ((uint16_t)spi_buf[18] << 8);
  blockD = spi_buf[19] + ((uint16_t)spi_buf[20] << 8);
}


/**------------------------------------------------------------------
 * @brief Queries the block statistic info of RDS decoder
 * @param none
 * @return none
 */
static void si468x_get_fm_rds_blockcount(void)
{
  spi_buf[0] = SI46XX_FM_RDS_BLOCKCOUNT;
  spi_buf[1] = 0x00;
  si468x_SpiMsg(spi_buf, 2);
  si468x_cts();
  si468x_responseN(10);
  blk_exp = spi_buf[5] + ((uint16_t)spi_buf[6] << 8); 
  blk_rec = spi_buf[7] + ((uint16_t)spi_buf[8] << 8); 
}


/**------------------------------------------------------------------
 * @brief Returns the Function revision information of the device.
 * @param none
 * @param[out] ver_major [8-bit]
 * @param[out] ver_minor [8-bit]
 * @param[out] ver_build [8-bit]
 * @return Command status
 */
uint8_t si468x_get_func_info(void)
{
  uint8_t status;
    
  spi_buf[0] = SI46XX_GET_FUNC_INFO;
  spi_buf[1] = 0x00;
  si468x_SpiMsg(spi_buf, 2);
  status = si468x_cts();
  si468x_responseN(8);
  ver_major = spi_buf[5];
  ver_minor = spi_buf[6];
  ver_build = spi_buf[7];
  return(status);
}


/**------------------------------------------------------------------
 * @brief Boots the image currently loaded in RAM
 * @param[out] status
 * @return Clear to send status
 */
uint8_t si468x_boot(uint16_t *stat)
{
  uint8_t result;
  
  spi_buf[0] = SI46XX_BOOT;
  spi_buf[1] = 0x00;

  si468x_SpiMsg(spi_buf, 2);
  result = si468x_cts();
  stat = spi_buf[4];
}


/**------------------------------------------------------------------
 * @brief Sets the value of a property
 * @param property (address) [16-bit]
 * @param[out] value [16-bit]
 * @return Clear to send status
 */
uint8_t si468x_get_property(uint16_t property, uint16_t *value)
{
  uint8_t result;
  
  spi_buf[0] = SI46XX_GET_PROPERTY;
  spi_buf[1] = 0x01;   // count
  spi_buf[2] = property & 0xFF;
  spi_buf[3] = (property >> 8) & 0xFF;
  
  si468x_SpiMsg(spi_buf, 4);
  result = si468x_cts();
  
  si468x_responseN(6);
  value = (spi_buf[6] << 8) + spi_buf[5];
  
  return(result);
}


/**------------------------------------------------------------------
 * @brief Sets the value of a property
 * @param property (address) [16-bit]
 * @param value [16-bit]
 * @return Clear to send status
 */
uint8_t si468x_set_property(uint16_t property, uint16_t value)
{
  spi_buf[0] = SI46XX_SET_PROPERTY;
  spi_buf[1] = 0x00;

  spi_buf[2] = property & 0xFF;
  spi_buf[3] = (property >> 8) & 0xFF;
  spi_buf[4] = value & 0xFF;
  spi_buf[5] = (value >> 8) & 0xFF;

  si468x_SpiMsg(spi_buf, 6);
  return(si468x_cts());
}



/**------------------------------------------------------------------
 * @brief Sets the value of a FLASH property
 * @param property (address) [16-bit]
 * @param value [16-bit]
 * @return Clear to send status
 */
uint8_t si468x_flash_set_property(uint16_t property, uint16_t value)
{
  spi_buf[0] = SI46XX_FLASH_LOAD;
  spi_buf[1] = 0x10;
  spi_buf[2] = 0x0;
  spi_buf[3] = 0x0;

  spi_buf[4] = property & 0xFF;    //SPI CLock
  spi_buf[5] = (property >> 8) & 0xFF;
  spi_buf[6] = value & 0xFF;
  spi_buf[7] = (value >> 8) & 0xFF;

  si468x_SpiMsg(spi_buf, 8);
  return(si468x_cts());
}


/**------------------------------------------------------------------
 * @brief Loads an image from external FLASH over secondary SPI bus
 * @param flash_addr start [32-bit]
 * @return Clear to send status
 */
uint8_t si468x_flash_load(uint32_t flash_addr)
{
  spi_buf[0] = SI46XX_FLASH_LOAD;
  spi_buf[1] = 0x00;
  spi_buf[2] = 0x00;
  spi_buf[3] = 0x00;

  spi_buf[4] = (flash_addr  & 0xff);
  spi_buf[5] = ((flash_addr >> 8)  & 0xff);
  spi_buf[6] = ((flash_addr >> 16)  & 0xff);
  spi_buf[7] = ((flash_addr >> 24) & 0xff);

  spi_buf[8] = 0x00;
  spi_buf[9] = 0x00;
  spi_buf[10] = 0x00;
  spi_buf[11] = 0x00;

  si468x_SpiMsg(spi_buf, 12);
  return(si468x_cts());
}


/**------------------------------------------------------------------
 * @brief Sets the DAB frequency table. The frequencies are in units 
 * of kHz
 * @param none
 * @return Clear to send status
 */
uint8_t si468x_dab_set_freq_list(void)
{
  uint8_t i;
  uint32_t  freq; 
  spi_buf[0] = SI46XX_DAB_SET_FREQ_LIST;
  spi_buf[1] = DAB_FREQS;
  spi_buf[2] = 0x00;
  spi_buf[3] = 0x00;

  for (i = 0; i < DAB_FREQS; i++)
  {
    freq = pgm_read_dword_near(dab_freq + i);
    spi_buf[4 + (i * 4)] = (freq  & 0xff);
    spi_buf[5 + (i * 4)] = ((freq >> 8)  & 0xff);
    spi_buf[6 + (i * 4)] = ((freq >> 16)  & 0xff);
    spi_buf[7 + (i * 4)] = ((freq >> 24) & 0xff);
  }
  si468x_SpiMsg(spi_buf, 4 + (i * 4));
  return(si468x_cts());
}


/**------------------------------------------------------------------
 * @brief Sets the DAB frequency table. The frequencies are in units 
 * of kHz
 * @param freq_index  frequency table index
 * @return Clear to send status
 */
static void si468x_dab_tune_freq(uint8_t freq_index)
{
  if ((freq_index >= 0) && (freq_index <= DAB_FREQS))
  {
    uint32_t timeout;
    spi_buf[0] = SI46XX_DAB_TUNE_FREQ;
    spi_buf[1] = 0x00;
    spi_buf[2] = freq_index;
    spi_buf[3] = 0x00;
    spi_buf[4] = 0x00;
    spi_buf[5] = 0x00;
    si468x_SpiMsg(spi_buf, 6);
    cmd_error = si468x_cts();

    timeout = 1000;
    do
    {
      _delay_ms(4);
      si468x_response();
      timeout--;
      if(timeout == 0)
      {
        cmd_error |= 0x80;
        break;
      }
    }
    while ((spi_buf[1] & 0x01) == 0); //STCINT
  }
}


/**------------------------------------------------------------------
 * @brief Returns status information about the digital radio and ensemble
 * @param none
 * @return none
 */
static void si468x_dab_digrad_status(void)
{
  spi_buf[0] = SI46XX_DAB_DIGRAD_STATUS;
  spi_buf[1] = 0x09; //Clear Interrupts: DIGRAD_ACK | STC_ACK
  si468x_SpiMsg(spi_buf, 2);
  si468x_cts();
}


/**------------------------------------------------------------------
 * @brief 
 * @param[out]  audio_bit_rate in kbps
 * @param[out]  audio_sample_rate in Nz
 * @param[out]  audio_mode 0: dual,1: mono,2: stereo,3: joint stereo
 * @param[out]  dabplus true/false
 * @return none
 */ 
static void si468x_dab_get_audio_info(void)
{
  spi_buf[0] = SI46XX_DAB_GET_AUDIO_INFO;
  spi_buf[1] = 0x00; 
  si468x_SpiMsg(spi_buf, 2);
  si468x_cts();
  si468x_responseN(20);

  audio_bit_rate = spi_buf[5] + ((uint16_t)spi_buf[6] << 8); 
  audio_sample_rate = spi_buf[7] + ((uint16_t)spi_buf[8] << 8); 
  audio_mode = spi_buf[9] & 0x03;
  dabplus = false;
  if ((spi_buf[9] & 0x0C) != 0) dabplus=true;
}


/**------------------------------------------------------------------
 * @brief Gets information about the sub-channel (service mode)
 * @param serviceID
 * @param compID
 * @param[out]  service_mode 0: AUDIO STREAM SERVICE
 *                           1: DATA STREAM SERVICE
 *                           2: FIDC SERVICE
 *                           3: MSC DATA PACKET SERVICE 
 *                           4: DAB+
 *                           5: DAB
 *                           6: FIC SERVICE
 *                           7: XPAD DATA 8 : NO MEDIA
 * @return none
 */ 
static void si468x_dab_get_subchan_info(uint32_t serviceID, uint32_t compID)
{
  spi_buf[0] = SI46XX_DAB_GET_SUBCHAN_INFO;
  spi_buf[1] = 0x00; 
  spi_buf[2] = 0x00; 
  spi_buf[3] = 0x00; 
  spi_buf[4] = (serviceID  & 0xff);
  spi_buf[5] = ((serviceID >> 8)  & 0xff);
  spi_buf[6] = ((serviceID >> 16)  & 0xff);
  spi_buf[7] = ((serviceID >> 24) & 0xff);

  spi_buf[8] = (compID  & 0xff);
  spi_buf[9] = ((compID >> 8)  & 0xff);
  spi_buf[10] = ((compID >> 16)  & 0xff);
  spi_buf[11] = ((compID >> 24) & 0xff);

  si468x_SpiMsg(spi_buf, 12);
  si468x_cts();
  si468x_responseN(12);

  service_mode = spi_buf[5]; 
}


/**------------------------------------------------------------------
 * @brief Gets information about the current ensemble
 * @param none
 * @return none
 */
void si468x_get_ensemble_info(void)
{
  spi_buf[0] = SI46XX_DAB_GET_ENSEMBLE_INFO;
  spi_buf[1] = 0x00;
  si468x_SpiMsg(spi_buf, 2);
  si468x_cts();
}


/**------------------------------------------------------------------
 * @brief Gets a service list of the ensemble
 * @param none
 * @return none
 */
static void si468x_get_digital_service_list(void)
{
  spi_buf[0] = SI46XX_GET_DIGITAL_SERVICE_LIST;
  spi_buf[1] = 0x00;
  si468x_SpiMsg(spi_buf, 2);
  si468x_cts();
}


/**------------------------------------------------------------------
 * @brief Gets information about the various events related to the DAB radio
 * @param none
 * @return none
 */
static void si468x_dab_get_event_status(void)
{
  spi_buf[0] = SI46XX_DAB_GET_EVENT_STATUS;
  spi_buf[1] = 0x00;
  si468x_SpiMsg(spi_buf, 2);
  si468x_cts();
}


/**------------------------------------------------------------------
 * @brief starts an audio or data service
 * @param serviceID : Service Number. This ID is found in the service list 
 *        returned by the GET_DIGITAL_SERVICE_LIST command
 * @param compID : Port/Program Number. This ID is found in the component 
 *        section of the service list returned by the GET_DIGITAL_SERVICE_LIST
 * @return none
 */
static void si468x_start_digital_service(uint32_t serviceID, uint32_t compID)
{
  spi_buf[0] = SI46XX_START_DIGITAL_SERVICE;
  spi_buf[1] = 0x00;
  spi_buf[2] = 0x00;
  spi_buf[3] = 0x00;
  spi_buf[4] = serviceID & 0xff;
  spi_buf[5] = (serviceID >> 8) & 0xff;
  spi_buf[6] = (serviceID >> 16) & 0xff;
  spi_buf[7] = (serviceID >> 24) & 0xff;
  spi_buf[8] = compID & 0xff;
  spi_buf[9] = (compID >> 8) & 0xff;
  spi_buf[10] = (compID >> 16) & 0xff;
  spi_buf[11] = (compID >> 24) & 0xff;
  si468x_SpiMsg(spi_buf, 12);
  si468x_cts();
}


/**------------------------------------------------------------------
 * @brief gets the ensemble time adjusted for the local time offset
 * @param time  structure dab_time
 * @return none
 */
void si468x_get_time(dab_time_t *time)
{
  spi_buf[0] = SI46XX_GET_TIME;
  spi_buf[1] = 0x00;  // local time 
  si468x_SpiMsg(spi_buf, 2);

  si468x_cts();
  si468x_responseN(12);

  time->year = spi_buf[5] + ((uint16_t)spi_buf[6] << 8);
  time->months = spi_buf[7];
  time->days = spi_buf[8];
  time->hours = spi_buf[9];
  time->minutes = spi_buf[10];
  time->seconds = spi_buf[11];
}


/**------------------------------------------------------------------
 * @brief set audio analog volume between 0..63
 * @param vol 
 * @return none
 */
void si468x_set_volume(uint8_t vol)
{
  si468x_set_property(0x0300, (vol & 0x3F));
}


/**------------------------------------------------------------------
 * @brief muite audio
 * @param mute
 *        0 : Do not mute audio outputs
 *        1 : Mute Left Audio Out.
 *        2 : Mute Right Audio Out.
 *        3 : Mute both Left and Right Audio Out
 * @return none
 */
void si468x_mute(uint8_t mute)
{
  si468x_set_property(0x0301, (mute & 0x03));
}


/**------------------------------------------------------------------
 * @brief wait max one second for clear to send status
 * @param none
 * @param[out] cmd_error  Command Error Code
 * @return command status
 */
uint8_t si468x_cts(void)
{
  uint32_t timeout;
  timeout = 1000;
  do
  {
    delay(4);
    si468x_response();
    timeout--;
    if(timeout == 0)
    {
      cmd_error = 0xFF;
      return(cmd_error);
    }
  }
  while ((spi_buf[1] & 0x80) == 0x00); // wait for CTS

  if ((spi_buf[1] & 0x40) == 0x40)
  {
    si468x_responseN(5);
    cmd_error = spi_buf[5];
  }
  return(cmd_error);
}


/**------------------------------------------------------------------
 * @brief check Data Interrupt indicator
 *        (valid reponse has to in spi_buffer)
 * @param intmask interrupt mask
 *        DACQINT = 0x20
 *        DSRVINT = 0x10
 *        RSQINT = 0x08
 *        RDSINT = 0x04
 *        ACFINT = 0x02
 *        STCINT = 0x01
 * @return true = active Interrupt
 */
bool si468x_check_int(uint8_t intmask)
{
  return((spi_buf[1] & intmask) == intmask);
}


/**------------------------------------------------------------------
 * @brief request and read four bytyes from si468x
 * @param none
 * @return none
 */
static void si468x_response(void)
{
  si468x_responseN(4);
}


/**------------------------------------------------------------------
 * @brief request and read N bytyes from si468x
 * @param N number of bytes to be read [16-bit]
 * @return nonr
 */
static void si468x_responseN(int len)
{
  int i;

  for (i = 0; i < len + 1; i++)
  {
    spi_buf[i] = 0;
  }

  si468x_SpiMsg(spi_buf, len + 1);
}


/**------------------------------------------------------------------
 * @brief Gets the maximum number of DAB frequency index
 * @param none
 * @return maximum number of DAB frequency index
 */
static uint8_t si468x_max_freq_index(void)
{
  return (DAB_FREQS); 
}
