#include <stdio.h>
#include <string.h>

// start addr of FM firmware in external FLASH
#define	FLASH_START_ADDR_FM		0x86000	// DAB+ Shield
//#define		FLASH_START_ADDR_FM		0x106000	// DAB+ Radio

#define DAB_MAX_SERVICEDATA_LEN	128
#define RDS_GROUP_0A	0
#define RDS_GROUP_0B	1
#define RDS_GROUP_2A	4
#define RDS_GROUP_2B	5
#define RDS_GROUP_4A	8

void (*_fm_callback)(uint16_t rdsdata);

// variables from si468x functions
extern uint16_t				fm_pi;					// Program ID
extern uint8_t				fm_pty;					// Program Type 
extern unsigned char 	spi_buf[];
extern uint8_t   			ver_major;
extern uint8_t   			ver_minor;
extern uint8_t   			ver_build;
extern uint8_t   			chip_rev;
extern uint8_t   			rom_id;
extern uint16_t  			part_no;
extern bool						valid;
extern uint16_t  			blockA;
extern uint16_t  			blockB;
extern uint16_t  			blockC;
extern uint16_t  			blockD;
extern uint16_t				blk_exp;		// RDS blocks expected
extern uint16_t 			blk_rec;		// RDS blocks received

// internal variables
char			fm_ps[9];							// FM Program Service Name
uint16_t	rdsdata;							// FM RDS data
char 			service_data[DAB_MAX_SERVICEDATA_LEN];	// service data buffer = Radio Text
uint8_t		last_text_ab_state;		//		
uint8_t		dab_status;						// status of si468x command
uint16_t	value;								// value of boot

//-----------------------------------------------------------------------------------------
// initialize FM mode
// return  status 
//-----------------------------------------------------------------------------------------
uint8_t fm_init()
{
	if (dab_status = si468x_reset() != 0) return(dab_status);
	if (dab_status = si468x_load_init() != 0) return(dab_status);
	if (dab_status = si468x_host_load() != 0) return(dab_status);
	if (dab_status = si468x_load_init() != 0) return(dab_status);
	if (dab_status = si468x_flash_set_property(0x0001, 5000) != 0) return(dab_status); 
	if (dab_status = si468x_flash_load(FLASH_START_ADDR_FM) != 0) return(dab_status); 
	if (dab_status = si468x_boot(&value) != 0) return(dab_status); 
	if (dab_status = si468x_set_property(0x0000, 0x0004) != 0) return(dab_status);	// INT_CTL_ENABLE RDS_INT 
	if (dab_status = si468x_set_property(0x0001, 0x0004) != 0) return(dab_status);	// INT_CTL_REPEAT RDS_INT
	if (dab_status = si468x_set_property(0x3900, 0x0001) != 0) return(dab_status);	// FM_AUDIO_DE_EMPHASIS
	if (dab_status = si468x_set_property(0x3C00, 0x0001) != 0) return(dab_status);  // FM_RDS_INTERRUPT_SOURCE
	if (dab_status = si468x_set_property(0x3C01, 0x0010) != 0) return(dab_status);  // FM_RDS_INTERRUPT_FIFO_COUNT
	if (dab_status = si468x_set_property(0x3C02, 0x0001) != 0) return(dab_status);  // FM_RDS_CONFIG
	return(dab_status);
}

//-----------------------------------------------------------------------------------------
// tune to FM frequewncy freq
// vol  0..63
//-----------------------------------------------------------------------------------------
void fm_tune_freq(uint16_t freq)
{
  uint8_t i;

  si468x_fm_tune_freq(freq);
  for(i=0; i<9; i++) {fm_ps[i] = '\0';}
  for(i=0; i<65; i++) {service_data[i] = '\0';}
  fm_pi = 0;
	fm_pty = 0;
}

//-----------------------------------------------------------------------------------------
// seek for next valid FM station and share signal quality values
// seekup  direction up true/false
// wrap    continue at band limit true/false
// return  tuned station valid true/false
//-----------------------------------------------------------------------------------------
bool fm_seek(uint8_t seekup, uint8_t wrap)
{
  uint8_t i;

	si468x_fm_seek(seekup, wrap);
	si468x_fm_rsq_status();
	for(i=0; i<9; i++) {fm_ps[i] = '\0';}
  for(i=0; i<65; i++) {service_data[i] = '\0';}
  fm_pi = 0;
	fm_pty = 0;
	if (valid) fm_status();
	return valid;
}

//-----------------------------------------------------------------------------------------
// Scan FM band and report valid stations 
//-----------------------------------------------------------------------------------------
void fm_scan(void)
{
  uint16_t startfreq = freq;
   
  si468x_mute(3);
  fm_tune_freq((uint16_t)8750);
  while(fm_seek(1, 0) == true);
  si468x_fm_tune_freq(startfreq);
  si468x_mute(0);
}

//-----------------------------------------------------------------------------------------
// Displays the tuned FM frequency and the signal quality values RSSI and SNR
//-----------------------------------------------------------------------------------------
void fm_status(void)
{
  char buffer[70];

  si468x_fm_rsq_status();
  si468x_get_fm_rds_blockcount();
  Serial.print(F("Freq="));
  sprintf(buffer, "%3u.", (uint16_t)freq / 100);
  Serial.print(buffer);
  sprintf(buffer, "%1u", (uint16_t)(freq % 100)/10);
  Serial.print(buffer);
  Serial.print(F("MHz RSSI="));     
  sprintf(buffer,"%d",rssi);
  Serial.print(buffer);
  Serial.print(F(" SNR="));
  sprintf(buffer,"%d ",snr);
  // Serial.print(buffer); 
  // Serial.print(F(" exp="));
  // sprintf(buffer,"%u",blk_exp);
  // Serial.print(buffer);  
  // Serial.print(F(" rec="));
  // sprintf(buffer,"%u ",blk_rec);
  // Serial.print(buffer);    
  sprintf(buffer,"%02u.%02u.%04u ", localtime.days,localtime.months,localtime.year);
  Serial.print(buffer);
  sprintf(buffer,"%02u:%02u", localtime.hours,localtime.minutes);
  Serial.print(buffer);
  Serial.print(F(" PI="));
  sprintf(buffer,"%04x",fm_pi);
  Serial.print(buffer);
  Serial.print(F(" PTY="));
  sprintf(buffer,"%u",fm_pty);
  Serial.print(buffer);
  Serial.print(F(" PS="));
  sprintf(buffer,"%s",fm_ps);
  Serial.print(buffer);
  Serial.print(F(" SD="));
  sprintf(buffer,"%s\n",service_data);
  Serial.print(buffer);    
}

//-----------------------------------------------------------------------------------------
// RDS Data Interrupt indicator.
// if RDSINT active then call _fm_callback
//-----------------------------------------------------------------------------------------
void fm_set_callback(void (*ServiceData)(void))
{
	_fm_callback = ServiceData;
}

//-----------------------------------------------------------------------------------------
// RDS Data Interrupt indicator.
// if RDSINT active then call _fm_callback
//-----------------------------------------------------------------------------------------
void fm_data_service(void) 
{
	si468x_response();
	//RDSINT active ?
	if (si468x_check_int(0x04))  
	{
		si468x_get_fm_rds_status();
		rdsdata = fm_decode_rds_group(blockA, blockB, blockC, blockD);
		if(rdsdata != 0)
			_fm_callback(rdsdata);
	}
}

//-----------------------------------------------------------------------------------------
// Decode received RDS data snippets
// param   blockA, blockB, blockC, blockD from RDS status call
// return  0x0000  nothing new
//         0x0001  new program name 
//         0x0004  new servie data
//         0x0010  new time update
//-----------------------------------------------------------------------------------------
uint16_t fm_decode_rds_group(uint16_t blockA, uint16_t blockB, uint16_t blockC, uint16_t blockD)
{
	(void)blockA;	//not used

	uint8_t ret, i;
	uint8_t (*rt_buffer)[64] = (uint8_t (*)[64])service;	//Reuse the DAB service memory
	uint8_t (*ps_buffer)[8] = (uint8_t (*)[8])ensemble_name;		//Reuse the DAB service data memory
	uint8_t	rds_group;
	uint8_t offset;
	ret = 0;

	rds_group = (blockB >> 11);
	switch(rds_group)
	{
	// program service name
	case RDS_GROUP_0A:
	case RDS_GROUP_0B:
		offset = blockB & 0x03;
		//copy last data into last buffer
		ps_buffer[1][(offset * 2)+0] = ps_buffer[0][(offset * 2)+0];
		ps_buffer[1][(offset * 2)+1] = ps_buffer[0][(offset * 2)+1];
		//load new data into current buffer
		ps_buffer[0][(offset * 2)+0] = blockD >> 8;
		ps_buffer[0][(offset * 2)+1] = blockD;
		if(offset == 3)
		{
			//look for a difference between consecative text
			for(i=0;i<8;i++)
			{
				if(ps_buffer[0][i] != ps_buffer[1][i])
					break;
			}
			//if no difference found check to see if new Radio Text
			if(i==8)
			{
				for(i=0;i<8;i++)
				{
					if(fm_ps[i] != ps_buffer[0][i])
						break;
				}
				//new Radio Text update buffer
				if(i!=8)
				{
					for(i=0;i<8;i++)
						fm_ps[i] = ps_buffer[0][i];
					ret |= 0x0001;
					fm_ps[8] = '\0';
				}
			}				
		}
		fm_ps[8] = '\0';  // Program Service Name
		break;

	// Radio Text
	case RDS_GROUP_2A:
	case RDS_GROUP_2B:
	{
		uint8_t offset;
		uint8_t i;
		uint8_t text_ab_state;
		uint8_t same;
		uint8_t text_size;
		uint8_t first_zero;

		text_ab_state = (blockB & 0x0010) ? 0x80 : 0x00;
		text_ab_state += rds_group;

		//the Group or Text A/B has changed to clear the buffers.
		if(text_ab_state != last_text_ab_state)
		{
			for(i=0;i<64;i++)
			{
				rt_buffer[0][i] = 0;
				rt_buffer[1][i] = 0;
			}
		}
		last_text_ab_state = text_ab_state;

		if(rds_group == RDS_GROUP_2A)
		{
			offset = (blockB & 0xf) * 4;

			rt_buffer[1][offset + 0] = rt_buffer[0][offset + 0];
			rt_buffer[1][offset + 1] = rt_buffer[0][offset + 1];
			rt_buffer[1][offset + 2] = rt_buffer[0][offset + 2];
			rt_buffer[1][offset + 3] = rt_buffer[0][offset + 3];

			rt_buffer[0][offset + 0] = (blockC >> 8)   ? (blockC >> 8)   : 0x20;
			rt_buffer[0][offset + 1] = (blockC & 0xff) ? (blockC & 0xff) : 0x20;
			rt_buffer[0][offset + 2] = (blockD >> 8)   ? (blockD >> 8)   : 0x20;
			rt_buffer[0][offset + 3] = (blockD & 0xff) ? (blockD & 0xff) : 0x20;

			//Are there differences, if so clear and start from scratch.
			if(((rt_buffer[1][offset + 0]) && (rt_buffer[1][offset + 0] != rt_buffer[0][offset + 0])) ||
				((rt_buffer[1][offset + 1]) && (rt_buffer[1][offset + 1] != rt_buffer[0][offset + 1])) ||
				((rt_buffer[1][offset + 2]) && (rt_buffer[1][offset + 2] != rt_buffer[0][offset + 2])) ||
				((rt_buffer[1][offset + 3]) && (rt_buffer[1][offset + 3] != rt_buffer[0][offset + 3])))
			{
				for(i=0;i<64;i++)
				{
					rt_buffer[0][i] = 0;
					rt_buffer[1][i] = 0;
				}           
				rt_buffer[0][offset + 0] = (blockC >> 8)   ? (blockC >> 8)   : 0x20;
				rt_buffer[0][offset + 1] = (blockC & 0xff) ? (blockC & 0xff) : 0x20;
				rt_buffer[0][offset + 2] = (blockD >> 8)   ? (blockD >> 8)   : 0x20;
				rt_buffer[0][offset + 3] = (blockD & 0xff) ? (blockD & 0xff) : 0x20;
			}
		}
		else
		{
			offset = (blockB & 0x0F) * 2;

			rt_buffer[1][offset + 0] = rt_buffer[0][offset + 0];
			rt_buffer[1][offset + 1] = rt_buffer[0][offset + 1];
			//load new data into current buffer
			rt_buffer[0][offset + 0] = (blockD >> 8)   ? (blockD >> 8)   : 0x20;
			rt_buffer[0][offset + 1] = (blockD & 0xff) ? (blockD & 0xff) : 0x20;
			//Are there differences, if so clear and start from scratch.
			if(((rt_buffer[1][offset + 0]) && (rt_buffer[1][offset + 0] != rt_buffer[0][offset + 0])) ||
				((rt_buffer[1][offset + 1]) && (rt_buffer[1][offset + 1] != rt_buffer[0][offset + 1])))
			{
				for(i=0;i<64;i++)
				{
					rt_buffer[0][i] = 0;
					rt_buffer[1][i] = 0;
				}           
				rt_buffer[0][offset + 0] = (blockD >> 8)   ? (blockD >> 8)   : 0x20;
				rt_buffer[0][offset + 1] = (blockD & 0xff) ? (blockD & 0xff) : 0x20;
			}
		}

		same = 1;
		offset = 0;
		text_size = 0;
		first_zero = 0xff;

		while (same && (offset < 64))
		{
			if(rt_buffer[1][offset] != rt_buffer[0][offset])
				same = 0;
			if(rt_buffer[0][offset] > 0)
			{
				text_size = offset;
			}
			else if(first_zero == 0xff)
			{
				first_zero = offset;
			}
			offset++;
		}
		if(first_zero < text_size)
		{
			same = 0;
		}

		if(same)
		{
			offset = 0;                             
			while (same && (offset < 64))
			{
				if(service_data[offset] != rt_buffer[0][offset])
					same = 0;
				offset++;
			}
			//
			if(same == 0)
			{
				for(i=0;i<64;i++)
					service_data[i] = rt_buffer[0][i];
				service_data[64] = '\0';
				ret |= 0x0004;	//Change of Status
			}
		}
	}
	break;

	// local time
	case RDS_GROUP_4A:
		uint32_t mjd;

		mjd = (blockB & 0x03);
		mjd <<= 15;
		mjd += ((blockC >> 1) & 0x7FFF); 

	    long J, C, Y, M;

		J = mjd + 2400001 + 68569;
		C = 4 * J / 146097;
		J = J - (146097 * C + 3) / 4;
		Y = 4000 * (J + 1) / 1461001;
		J = J - 1461 * Y / 4 + 31;
		M = 80 * J / 2447;
		localtime.days = J - 2447 * M / 80;
		J = M / 11;
		localtime.months = M + 2 - (12 * J);
		localtime.year = 100 * (C - 49) + Y + J;

		localtime.hours = ((blockD >> 12) & 0x0f);
		localtime.hours += ((blockC << 4) & 0x0010);
		localtime.minutes = ((blockD >> 6) & 0x3f);
		//LocalOffset = (blockD & 0x3f);
		ret |= 0x0010;
	break;
	}
	return ret;
}

//-----------------------------------------------------------------------------------------
// test FM function of DAB+ module
// freq = set tuning frequency
// return  successful true/false
//-----------------------------------------------------------------------------------------
bool fm_module_test(uint16_t freq)
{
  char buffer[32];

	dab_status = si468x_reset();
	if (dab_status != 0) {
		Serial.print(F(" Reset ERROR: "));
		sprintf(buffer, "%u\n", dab_status);
		Serial.print(buffer);    
	} else {
		Serial.print(F(" Reset PASS \n")); 

		dab_status = si468x_get_part_info();
		if (dab_status != 0) {
			Serial.print(F("Get Part Info ERROR: "));
			sprintf(buffer, "%u\n", dab_status);
			Serial.print(buffer);    
		} else {
			Serial.print(F(" Get Part Info PASS -> Chip Revision:"));
			sprintf(buffer, "%u, RomID:%u, Part No:%u\n", chip_rev, rom_id, part_no);
			Serial.print(buffer);   

			dab_status = si468x_load_init();
			if (dab_status != 0) {
				Serial.print(F(" Load Init ERROR: "));
				sprintf(buffer, "%u\n", dab_status);
				Serial.print(buffer);    
			} else {
				Serial.print(F(" Load Init PASS \n")); 

			  dab_status = si468x_host_load();
			  if (dab_status != 0) {
			    Serial.print(F(" Host Load ERROR: "));
					sprintf(buffer, "%d\n", dab_status);
					Serial.print(buffer);    
			   } else {
					Serial.print(F(" Host Load PASS \n")); 

					dab_status = si468x_load_init();
					if (dab_status != 0) {
								Serial.print(F(" Load Init ERROR: "));
			        	sprintf(buffer, "%u\n", dab_status);
			        	Serial.print(buffer);    
					} else {
			        	Serial.print(F(" Load Init PASS \n")); 

						//Set SPI Clock to 10 MHz
						dab_status = si468x_flash_set_property(0x0001, 5000);
						if (dab_status != 0) {
							Serial.print(F(" FLASH Set Property ERROR: "));
							sprintf(buffer, "%u\n", dab_status);
							Serial.print(buffer);    
						} else {
							Serial.print(F(" FLASH Set Property PASS \n")); 

							dab_status = si468x_flash_load(FLASH_START_ADDR_FM); 
							if (dab_status != 0) {
								Serial.print(F(" FLASH Load FM ERROR: "));
								sprintf(buffer, "%u\n", dab_status);
								Serial.print(buffer);    
							} else {
				        Serial.print(F(" FLASH Load FM PASS \n")); 
			            
								dab_status = si468x_boot(&value);
								if (dab_status != 0) {
									Serial.print(F(" Boot ERROR: "));
									sprintf(buffer, "%u -> Stat:%X\n", dab_status, value);
									Serial.print(buffer);    
								} else {
									Serial.print(F(" Boot PASS -> Stat: "));
									sprintf(buffer, "%X\n", value);
									Serial.print(buffer);  

									dab_status = si468x_get_func_info();
									if (dab_status != 0) {
										Serial.print(F(" Get Func ERROR: "));
										sprintf(buffer, "%u\n", dab_status);
										Serial.print(buffer);    
									} else {
										Serial.print(F(" Get Func PASS -> FM Ver:"));
										sprintf(buffer, "%u.%u.%u\n", ver_major, ver_minor, ver_build);
										Serial.print(buffer);  
			              
										dab_status = si468x_get_property(0x0201, &value);
										if (dab_status != 0) {
											Serial.print(F(" Get Property ERROR: "));
											sprintf(buffer, "%u\n", dab_status);
											Serial.print(buffer);    
										} else {
											Serial.print(F(" Get Property PASS -> 0x0201:"));
											sprintf(buffer, "%u\n", value);
											Serial.print(buffer);   

											//Set up INTB
											dab_status = si468x_set_property(0x0000, 0x0004);
											dab_status = si468x_set_property(0x3900, 0x0001);
											dab_status = si468x_set_property(0x3C00, 0x0001);
											dab_status = si468x_set_property(0x3C01, 0x0010);
											dab_status = si468x_set_property(0x3C02, 0x0001);
											if (dab_status != 0) {
												Serial.print(F(" Set Property ERROR: "));
												sprintf(buffer, "%u\n", dab_status);
												Serial.print(buffer);    
											} else {
												Serial.print(F(" Set Property PASS \n")); 
											}
			                
											dab_status = si468x_fm_tune_freq(freq);
											if (dab_status != 0) {
												Serial.print(F(" FM Tune Freq ERROR: "));
												sprintf(buffer, "%u\n", dab_status);
												Serial.print(buffer);    
											} else {
												Serial.print(F(" FM Tune Freq PASS \n"));     
												return(true);
 											}
 										}
 									}
 								}
 							}
 						}
 					}
 				}
 			}
 		}
 	}
 	return(false);
}
